import "engine" for Graphics, Input, Logger, Raycast, Resource, Scene, Window
import "Scripts/light" for Light
import "Scripts/cube" for Cube
import "Scripts/camera" for FPSCamera

// Configuration: set to true to use blend file assets, false to create programmatically
var USE_BLEND_ASSETS = true

class Game {
    construct new() {
        _blend = null
        _scene = null
        _fpsCamera = null
        _shader = null
        
        // Interactive objects
        _cube = null
        _light = null
    }

    init() {
        // Create fresh scene (important for hot-reload to avoid duplicates)
        _scene = Scene.new()
        // Load the default shader for rendering
        _shader = Graphics.loadShader("shaders/default")
        if (_shader) {
            Graphics.useShader(_shader)
            Logger.info("Loaded default shader")
        } else {
            Logger.error("Failed to load default shader")
        }
        
        // Load the blend file (for mesh data - still needed for the cube)
        _blend = Resource.loadBlend("Models/Cube.blend")
        if (!_blend) Fiber.abort("Failed to load Models/Cube.blend")
        Logger.info("Loaded blend: %(nodeCount) nodes")
        
        // List what's in the blend file
        for (i in 0..._blend.nodeCount) {
            var name = _blend.nodeName(i)
            var nodeId = _blend.nodeId(i)
            Logger.info("  Node %(i): name='%(name)' nodeId='%(nodeId)'")
        }

        // Instantiate the Cube (collection with class "Cube")
        var cubeRoot = _blend.instantiate("Cube", _scene)
        if (cubeRoot != null) {
            Logger.info("Instantiated Cube collection")
            // The mesh is on the child node with nodeId "model"
            if (cubeRoot.nodes.containsKey("model")) {
                _cube = Cube.new()
                _cube.init(cubeRoot.nodes["model"], null)
            }
        }

        // =========================================================================
        // CAMERA SETUP - Two approaches:
        // =========================================================================
        if (USE_BLEND_ASSETS) {
            // APPROACH 1: Use camera from Blender file
            // Camera is auto-created with Blender's settings (lens, clip planes, etc.)
            var cameraRoot = _blend.instantiate("GameCamera", _scene)
            if (cameraRoot != null) {
                Logger.info("Instantiated Camera from blend file")
                // List what nodes we got from instantiation
                for (nodeId in cameraRoot.nodes.keys) {
                    var node = cameraRoot.nodes[nodeId]
                    var npos = node.getPosition()
                    Logger.info("  InstantiatedNode: nodeId='%(nodeId)' name='%(node.name)' pos=(%(npos[0]),%(npos[1]),%(npos[2]))")
                }
                // Try to find camera by tag - check both 'camera' (nodeId) and 'Camera' (name)
                var blendCamera = _scene.findCameraByTag("camera")
                if (!blendCamera) blendCamera = _scene.findCameraByTag("Camera")
                if (!blendCamera) blendCamera = _scene.findCameraByTag("OBCamera")
                if (blendCamera) {
                    Logger.info("  Found camera with tag: %(blendCamera.getTag())")
                    var camNode = blendCamera.node
                    if (camNode) {
                        var nodePos = camNode.getPosition()
                        Logger.info("  Camera node position: (%(nodePos[0]),%(nodePos[1]),%(nodePos[2]))")
                        _fpsCamera = FPSCamera.fromCamera(_scene, blendCamera)
                        Logger.info("  Created FPSCamera, node pos check: %(_fpsCamera.position)")
                    } else {
                        Logger.error("  Camera has no node attached!")
                    }
                } else {
                    Logger.warn("  Could not find camera by tag - check camera nodeId in blend file")
                }
            }
        }
        
        // APPROACH 2: Create camera programmatically (fallback or standalone)
        if (_fpsCamera == null) {
            Logger.info("Creating camera programmatically")
            _fpsCamera = FPSCamera.new(_scene, "main")
            // _fpsCamera.setPosition(0, 0, 5)
            // Camera settings can be customized:
            // _fpsCamera.camera.setNearPlane(0.1)
            // _fpsCamera.camera.setFarPlane(1000)
            // _fpsCamera.camera.setFovYRadians(1.047) // ~60 degrees
        }
        _fpsCamera.moveSpeed = 5.0
        _fpsCamera.lookSensitivity = 0.003

        // =========================================================================
        // LIGHT SETUP - Two approaches:
        // =========================================================================
        if (USE_BLEND_ASSETS) {
            // APPROACH 1: Use light from Blender file
            // Light is auto-created with Blender's settings (type, color, energy, etc.)
            var lightRoot = _blend.instantiate("Light", _scene)
            if (lightRoot != null) {
                Logger.info("Instantiated Light from blend file")
                _light = _scene.findLightByTag("Light")
                if (_light) {
                    Logger.info("  Light type: %(_light.type), energy: %(_light.energy)")
                    Logger.info("  Light color: R=%(_light.r) G=%(_light.g) B=%(_light.b)")
                }
            }
        }
        
        // APPROACH 2: Create light programmatically (fallback or standalone)
        if (_light == null) {
            Logger.info("Creating light programmatically")
            _light = _scene.addLight()
            _light.setTag("main_light")
            // Attach to a node for positioning
            var lightNode = _scene.addNode("LightNode")
            lightNode.setPosition(4, 4, 4)
            _light.setNode(lightNode)
            // Configure light properties
            _light.type = 0          // 0=Point, 1=Sun, 2=Spot, 3=Hemi, 4=Area
            _light.setColor(1, 1, 1) // White light
            _light.energy = 1000     // Intensity
            _light.radius = 0.1      // Soft shadow radius
            Logger.info("  Created point light at (4, 4, 4)")
        }
        
        // Print controls
        Logger.info("")
        Logger.info("=== Controls ===")
        Logger.info("  Left-click + drag cube: Rotate cube")
        Logger.info("  Right-click + drag: Look around")
        Logger.info("  WASD / Arrow keys: Move camera")
        Logger.info("  Space: Move up")
        Logger.info("  Ctrl: Move down")
        Logger.info("================")
    }

    nodeCount { _blend != null ? _blend.nodeCount : 0 }

    update(dt) {
        // Update FPS camera (handles right-click look and WASD movement)
        _fpsCamera.update(dt)
        
        // Handle cube interaction (only when not looking with camera)
        if (!_fpsCamera.isLooking) {
            handleCubeInteraction()
        } else {
            // End cube drag if we start looking
            if (_cube != null) _cube.endDrag()
        }
    }
    
    handleCubeInteraction() {
        var mouseX = Input.mouseX()
        var mouseY = Input.mouseY()
        
        // Start drag on left click
        if (Input.mouseJustPressedLeft() && _cube != null) {
            var vpW = Window.getWidth()
            var vpH = Window.getHeight()
            var hit = Raycast.fromCamera(_scene, _fpsCamera.camera, mouseX, mouseY, vpW, vpH)
            _cube.tryStartDrag(hit, mouseX, mouseY)
        }
        
        // Update drag
        if (_cube != null && _cube.isDragging) {
            if (Input.mouseLeft()) {
                _cube.updateDrag(mouseX, mouseY)
            } else {
                _cube.endDrag()
            }
        }
    }

    draw() {
        Graphics.setViewProjectionEnabled(true)
        var aspect = Window.getWidth() / Window.getHeight()
        if (aspect < 0.01) aspect = 1
        
        // Use the camera's view matrix (now correctly computed by the engine)
        var view = _fpsCamera.camera.getViewMatrix()
        var proj = _fpsCamera.camera.getProjectionMatrix(aspect)
        Graphics.setViewMatrix(view)
        Graphics.setProjectionMatrix(proj)
        
        _scene.draw()
    }

    quit() {}
}
