// Plume3D Wren script - Interactive Cube component
// Supports click-and-drag rotation

import "engine" for Input, Logger, Math, Raycast, Window

// Drag sensitivity for rotation
var DRAG_SENSITIVITY = 0.01

class Cube {
    construct new() {
        _node = null
        _config = null
        
        // Drag state
        _isDragging = false
        _dragStartRotX = 0.0
        _dragStartRotY = 0.0
        _dragStartMouseX = 0.0
        _dragStartMouseY = 0.0
    }

    // Called by engine after instantiation
    init(node, config) {
        _node = node
        _config = config
        Logger.info("Cube initialized: %(node.name)")
    }

    node { _node }
    config { _config }
    isDragging { _isDragging }
    
    // Check if a raycast hit this cube and start dragging
    // Returns true if drag started
    tryStartDrag(hit, mouseX, mouseY) {
        if (hit == null || _node == null) return false
        
        var hitNodeName = hit.getNodeName()
        if (hitNodeName != _node.name) return false
        
        _isDragging = true
        var rot = _node.getRotation()
        _dragStartRotX = rot[0]
        _dragStartRotY = rot[1]
        _dragStartMouseX = mouseX
        _dragStartMouseY = mouseY
        
        Logger.info("Started dragging: %(_node.name)")
        return true
    }
    
    // Update drag rotation
    // Call this every frame while dragging
    updateDrag(mouseX, mouseY) {
        if (!_isDragging || _node == null) return
        
        var deltaX = mouseX - _dragStartMouseX
        var deltaY = mouseY - _dragStartMouseY
        var rot = _node.getRotation()
        
        // Horizontal drag rotates around Y, vertical drag rotates around X
        var newRotY = _dragStartRotY + deltaX * DRAG_SENSITIVITY
        var newRotX = _dragStartRotX + deltaY * DRAG_SENSITIVITY
        
        _node.setRotation(newRotX, newRotY, rot[2])
    }
    
    // End the drag
    endDrag() {
        if (_isDragging) {
            _isDragging = false
            Logger.info("Stopped dragging")
        }
    }
    
    // Set rotation directly (in radians)
    setRotation(x, y, z) {
        if (_node != null) {
            _node.setRotation(x, y, z)
        }
    }
    
    // Get current rotation
    getRotation() {
        if (_node != null) {
            return _node.getRotation()
        }
        return [0, 0, 0]
    }
    
    // Rotate by delta (in radians)
    rotate(dx, dy, dz) {
        if (_node != null) {
            var rot = _node.getRotation()
            _node.setRotation(rot[0] + dx, rot[1] + dy, rot[2] + dz)
        }
    }
}
