// Plume3D Wren script - FPSCamera helper class
// Encapsulates first-person camera controls with mouse look and WASD movement

import "engine" for Input, Logger, Math, Scene

// FPS-style camera controller
// Usage:
//   var fps = FPSCamera.new(scene, "main")
//   fps.setPosition(0, 0, 5)
//   // In update:
//   fps.update(dt)
//   // In draw:
//   var view = fps.camera.getViewMatrix()
class FPSCamera {
    // Create new camera and node from scratch
    construct new(scene, tag) {
        _scene = scene
        _camera = scene.addCamera()
        _camera.setTag(tag)
        _node = scene.addNode("FPSCameraNode")
        _camera.setNode(_node)
        initState_()
    }
    
    // Wrap an existing camera (e.g. from a .blend file)
    // Usage: var fps = FPSCamera.fromCamera(scene, scene.findCameraByTag("Camera"))
    construct fromCamera(scene, camera) {
        _scene = scene
        _camera = camera
        _node = camera.node
        
        // Convert Blender's arbitrary rotation to FPS-compatible yaw/pitch
        // This preserves the view direction but removes roll for level horizon
        convertToFpsRotation_()
        
        initState_()
    }
    
    // Convert the node's current rotation to FPS-style yaw/pitch
    // This extracts the view direction and converts it to yaw/pitch with zero roll
    convertToFpsRotation_() {
        // Get the current forward direction (accounts for full rotation including roll)
        var forward = _node.getForward()
        
        // Calculate yaw from forward vector (rotation around Y axis)
        // forward.x = -sin(yaw) * cos(pitch), forward.z = -cos(yaw) * cos(pitch)
        // yaw = atan2(-forward.x, -forward.z)
        var yaw = Math.atan2(-forward[0], -forward[2])
        
        // Calculate pitch from forward vector (rotation around X axis)
        // forward.y = sin(pitch)
        // pitch = asin(forward.y), clamped to avoid issues at poles
        var fy = forward[1]
        if (fy > 0.999) fy = 0.999
        if (fy < -0.999) fy = -0.999
        var pitch = Math.asin(fy)
        
        // Apply the FPS-compatible rotation (zero roll)
        _node.setRotation(pitch, yaw, 0)
    }
    
    // Initialize common state
    initState_() {
        // Settings
        _lookSensitivity = 0.003
        _moveSpeed = 5.0
        
        // Input state
        _isLooking = false
        _lastMouseX = 0.0
        _lastMouseY = 0.0
    }
    
    // Accessors
    camera { _camera }
    node { _node }
    
    // Orientation (delegates to node)
    yaw { _node.yaw }
    yaw=(v) { _node.yaw = v }
    pitch { _node.pitch }
    pitch=(v) { _node.pitch = v }
    roll { _node.roll }
    roll=(v) { _node.roll = v }
    
    // Settings
    lookSensitivity { _lookSensitivity }
    lookSensitivity=(v) { _lookSensitivity = v }
    moveSpeed { _moveSpeed }
    moveSpeed=(v) { _moveSpeed = v }
    
    // Position (delegates to node)
    position { _node.getPosition() }
    setPosition(x, y, z) { _node.setPosition(x, y, z) }
    
    // Look at a target point (delegates to node)
    lookAt(x, y, z) { _node.lookAt(x, y, z) }
    
    // Update camera based on input
    update(dt) {
        handleMouseLook()
        handleMovement(dt)
    }
    
    // Handle right-click mouse look
    handleMouseLook() {
        var mouseX = Input.mouseX()
        var mouseY = Input.mouseY()
        
        if (Input.mouseJustPressedRight()) {
            _isLooking = true
            _lastMouseX = mouseX
            _lastMouseY = mouseY
        }
        
        if (!Input.mouseRight()) {
            _isLooking = false
        }
        
        if (_isLooking) {
            var deltaX = mouseX - _lastMouseX
            var deltaY = mouseY - _lastMouseY
            _lastMouseX = mouseX
            _lastMouseY = mouseY
            
            if (deltaX.abs > 0.01 || deltaY.abs > 0.01) {
                // For FPS-style mouse look:
                // - Horizontal mouse movement rotates around world Y axis (yaw)
                // - Vertical mouse movement rotates around local X axis (pitch)
                
                // Update yaw (world Y rotation)
                _node.yaw = _node.yaw - deltaX * _lookSensitivity
                
                // Update pitch (local X rotation) - clamp to prevent flipping
                var newPitch = _node.pitch - deltaY * _lookSensitivity
                var maxPitch = Math.pi() / 2 - 0.01
                if (newPitch > maxPitch) newPitch = maxPitch
                if (newPitch < -maxPitch) newPitch = -maxPitch
                _node.pitch = newPitch
            }
        }
    }
    
    // Handle WASD + Space/Ctrl movement
    handleMovement(dt) {
        var speed = _moveSpeed * dt
        // Use node's built-in direction methods
        var forward = _node.getForward()
        var right = _node.getRight()
        var pos = _node.getPosition()
        
        // WASD movement
        if (Input.key("w") || Input.key("up")) {
            pos[0] = pos[0] + forward[0] * speed
            pos[1] = pos[1] + forward[1] * speed
            pos[2] = pos[2] + forward[2] * speed
        }
        if (Input.key("s") || Input.key("down")) {
            pos[0] = pos[0] - forward[0] * speed
            pos[1] = pos[1] - forward[1] * speed
            pos[2] = pos[2] - forward[2] * speed
        }
        if (Input.key("a") || Input.key("left")) {
            pos[0] = pos[0] - right[0] * speed
            pos[2] = pos[2] - right[2] * speed
        }
        if (Input.key("d") || Input.key("right")) {
            pos[0] = pos[0] + right[0] * speed
            pos[2] = pos[2] + right[2] * speed
        }
        
        // Up/Down movement (world space)
        if (Input.key("space")) {
            pos[1] = pos[1] + speed
        }
        if (Input.modCtrl()) {
            pos[1] = pos[1] - speed
        }
        
        _node.setPosition(pos[0], pos[1], pos[2])
    }
    
    // Check if camera is currently in look mode
    isLooking { _isLooking }
}

// Legacy GameCamera for backward compatibility
class GameCamera {
    construct new() {
        _node = null
        _config = null
        _camera = null
    }

    init(node, config) {
        _node = node
        _config = config
        Logger.info("GameCamera initialized: %(node.name)")
    }

    node { _node }
    config { _config }
    camera { _camera }
    camera=(c) { _camera = c }
}
