import "engine" for Audio, Config, Input, Gui, Window, Logger, Math

var TUNE_MIN = 88.0
var TUNE_MAX = 108.0
var TUNE_WIDTH = 2.5   // Hz from perfect frequency before full static
var TUNE_STEP = 0.1

class Game {
  construct new() {
    _frequency
    _channels
    _sources
    _staticSource
    _currentStationName
  }

  init() {
    _frequency = 98.0
    _channels = []
    _sources = []
    _staticSource = null

    var config = Config.load("channels.toml")
    var channelList = config["channels"]
    if (channelList == null) channelList = []
    for (ch in channelList) {
      _channels.add(ch)
      var src = Audio.newSource(ch["path"])
      if (src != null) {
        src.setLooping(true)
        _sources.add(src)
      } else {
        var pathStr = ch["path"]
        Logger.warn("Audio: failed to load '%(pathStr)' - check file exists under mounted app dir")
        _sources.add(null)
      }
    }

    _staticSource = Audio.newSource("sounds/static.wav")
    if (_staticSource != null) _staticSource.setLooping(true)
    if (_staticSource == null) Logger.warn("sounds/static.wav missing - no static effect")
  }

  update(dt) {
    if (Input.key("Left"))  _frequency = Math.max(TUNE_MIN, _frequency - TUNE_STEP)
    if (Input.key("Right")) _frequency = Math.min(TUNE_MAX, _frequency + TUNE_STEP)

    var closestIdx = -1
    var closestDist = 999.0
    var i = 0
    for (ch in _channels) {
      var dist = Math.abs(ch["frequency"] - _frequency)
      if (dist < closestDist) {
        closestDist = dist
        closestIdx = i
      }
      i = i + 1
    }

    var signalStrength = 0.0
    if (closestIdx >= 0 && closestDist <= TUNE_WIDTH) {
      signalStrength = 1.0 - (closestDist / TUNE_WIDTH)
    }
    var staticGain = 1.0 - signalStrength

    for (i in 0..._sources.count) {
      var src = _sources[i]
      if (src == null) continue
      if (i == closestIdx && signalStrength > 0.01) {
        src.setVolume(signalStrength)
        if (!Audio.isPlaying(src)) Audio.play(src)
      } else {
        Audio.stop(src)
      }
    }
    if (_staticSource != null) {
      if (staticGain > 0.01) {
        _staticSource.setVolume(staticGain)
        if (!Audio.isPlaying(_staticSource)) Audio.play(_staticSource)
      } else {
        Audio.stop(_staticSource)
      }
    }

    _currentStationName = ""
    if (closestIdx >= 0 && signalStrength > 0.5) {
      _currentStationName = _channels[closestIdx]["name"]
    }

    var width = Window.getWidth()
    var height = Window.getHeight()

    if (Gui.beginWindow("Radio", 0, 0, width, height)) {
      Gui.layoutRowDynamic(24, 1)
      Gui.label("Tune: Left / Right arrow keys")
      Gui.layoutRowDynamic(20, 1)
      Gui.label("FM %(_frequency) MHz")
      Gui.layoutRowDynamic(22, 1)
      Gui.label(_currentStationName != "" ? "Now playing: %(_currentStationName)" : "--- static ---")
      Gui.layoutRowDynamic(18, 1)
      Gui.label("Stations (from channels.toml):")
      for (ch in _channels) {
        Gui.layoutRowDynamic(18, 1)
        var f = ch["frequency"]
        var n = ch["name"]
        Gui.label("  %(f) - %(n)")
      }
      Gui.endWindow()
    }
  }

  draw() {}
  quit() {}
}
