import "engine" for Engine, Graphics, Gui, Input, Logger, Raycast, Resource, Scene, Window
import "Scripts/camera" for FPSCamera
import "Scripts/light" for Light
import "Scripts/radio" for Radio

class Game {
  construct new() {
    _scene = null
    _fpsCamera = null
    _light = null
    _blend = null
    _radio = null
    _hitNodeName = ""
    _screenshotTimer = 2.0  // Auto-screenshot after 2 seconds
    _screenshotTaken = false
  }

  init() {
    // Create fresh scene
    _scene = Scene.new()

    // Load shader
    var shader = Graphics.loadShader("shaders/default")
    if (shader) {
      Graphics.useShader(shader)
      Logger.info("Loaded default shader")
    } else {
      Logger.error("Failed to load default shader")
    }

    // Try to load compressed blend first, fall back to uncompressed
    _blend = Resource.loadBlend("Models/Radio_V3.blend")
    if (!_blend) {
      Fiber.abort("Failed to load Radio blend file")
    }
    
    Logger.info("Loaded blend file with %(_blend.nodeCount) nodes:")
    for (i in 0..._blend.nodeCount) {
      var name = _blend.nodeName(i)
      var nodeId = _blend.nodeId(i)
      Logger.info("  Node %(i): name='%(name)' nodeId='%(nodeId)'")
    }
    Logger.info("Blend classes: %(_blend.classes)")

    // =========================================================================
    // RADIO SETUP
    // =========================================================================
    var radioRoot = _blend.instantiate("Radio", _scene)
    if (radioRoot != null) {
      Logger.info("Instantiated Radio collection")
      var nodeList = []
      for (k in radioRoot.nodes.keys) { nodeList.add(k) }
      Logger.info("  Instance nodes (%(nodeList.count)): %(nodeList)")
      
      // Create Radio component and initialize with the instantiated nodes
      _radio = Radio.new()
      _radio.init(radioRoot, null)
    } else {
      Logger.warn("Failed to instantiate Radio - check blend file has 'Radio' class")
    }

    // =========================================================================
    // CAMERA SETUP
    // =========================================================================
    var cameraRoot = _blend.instantiate("GameCamera", _scene)
    if (cameraRoot != null) {
      Logger.info("Instantiated Camera from blend file")
      // Try to find camera by various tag names
      var blendCamera = _scene.findCameraByTag("camera")
      if (!blendCamera) blendCamera = _scene.findCameraByTag("Camera")
      if (!blendCamera) blendCamera = _scene.findCameraByTag("OBCamera")
      
      if (blendCamera && blendCamera.node) {
        var nodePos = blendCamera.node.getPosition()
        Logger.info("  Camera position: (%(nodePos[0]), %(nodePos[1]), %(nodePos[2]))")
        _fpsCamera = FPSCamera.fromCamera(_scene, blendCamera)
        Logger.info("  Created FPSCamera from blend")
      }
    }
    
    // Fallback: create camera programmatically
    if (_fpsCamera == null) {
      Logger.info("Creating camera programmatically")
      _fpsCamera = FPSCamera.new(_scene, "main")
      _fpsCamera.setPosition(0, 1, 3)
    }
    _fpsCamera.moveSpeed = 3.0
    _fpsCamera.lookSensitivity = 0.003

    // =========================================================================
    // LIGHT SETUP
    // =========================================================================
    var lightRoot = _blend.instantiate("Light", _scene)
    if (lightRoot != null) {
      Logger.info("Instantiated Light from blend file")
      _light = _scene.findLightByTag("light")
      if (!_light) _light = _scene.findLightByTag("Light")
      if (_light) {
        Logger.info("  Light type: %(_light.type), energy: %(_light.energy)")
      }
    }
    
    // Fallback: create light programmatically
    if (_light == null) {
      Logger.info("Creating light programmatically")
      _light = _scene.addLight()
      _light.setTag("main_light")
      var lightNode = _scene.addNode("LightNode")
      lightNode.setPosition(4, 4, 4)
      _light.setNode(lightNode)
      _light.type = 0          // Point light
      _light.setColor(1, 1, 1)
      _light.energy = 1000
      _light.radius = 0.1
    }

    // Print controls
    Logger.info("")
    Logger.info("=== Controls ===")
    Logger.info("  Power: click power button")
    Logger.info("  Body: left-click + drag to rotate")
    Logger.info("  Knobs: drag to tune frequency / volume")
    Logger.info("  Right-click + drag: Look around")
    Logger.info("  WASD / Arrow keys: Move camera")
    Logger.info("================")
  }

  update(dt) {
    if (Input.keyJustPressed("f12")) {
      Engine.screenshot("debug_screenshot.png")
    }
    
    // Auto-screenshot after timer
    if (!_screenshotTaken && _screenshotTimer > 0) {
      _screenshotTimer = _screenshotTimer - dt
      if (_screenshotTimer <= 0) {
        Engine.screenshot("debug_screenshot.png")
        _screenshotTaken = true
        Logger.info("Auto-screenshot taken")
      }
    }

    // Update FPS camera (handles right-click look and WASD movement)
    _fpsCamera.update(dt)

    // Update radio state (audio, animations)
    if (_radio != null) {
      _radio.update(dt)
    }

    var mouseX = Input.mouseX()
    var mouseY = Input.mouseY()
    var vpW = Window.getWidth()
    var vpH = Window.getHeight()

    // End drag on mouse release
    if (!Input.mouseLeft() && _radio != null && _radio.isDragging) {
      _radio.endDrag()
    }

    // Handle mouse clicks (only when not looking with camera)
    if (Input.mouseJustPressedLeft() && !_fpsCamera.isLooking) {
      var hit = Raycast.fromCamera(_scene, _fpsCamera.camera, mouseX, mouseY, vpW, vpH)
      if (hit != null) {
        _hitNodeName = hit.getNodeName()
        
        // Try radio interaction first
        if (_radio != null) {
          _radio.tryStartInteraction(hit, mouseX, mouseY)
        }
      } else {
        _hitNodeName = ""
      }
    }

    // Update radio drag if active
    if (Input.mouseLeft() && _radio != null && _radio.isDragging) {
      _radio.updateDrag(mouseX, mouseY)
    }
  }

  draw() {
    Graphics.setViewProjectionEnabled(true)
    var aspect = Window.getWidth() / Window.getHeight()
    if (aspect < 0.01) aspect = 1
    var view = _fpsCamera.camera.getViewMatrix()
    var proj = _fpsCamera.camera.getProjectionMatrix(aspect)
    Graphics.setViewMatrix(view)
    Graphics.setProjectionMatrix(proj)
    _scene.draw()

    // GUI overlay
    var w = Window.getWidth()
    var h = Window.getHeight()
    if (Gui.beginWindow("Radio 3D", 0, 0, w * 0.25, h * 0.3)) {
      Gui.layoutRowDynamic(22, 1)
      Gui.label("Click power button to toggle")
      Gui.label("Drag body to rotate")
      Gui.label("Drag knobs to tune/volume")
      
      Gui.layoutRowDynamic(20, 1)
      if (_radio != null) {
        Gui.label("Power: %(_radio.powerOn)")
        Gui.label("FM %(_radio.frequency) MHz")
        Gui.label("Volume: %((_radio.volume * 100).floor)\%")
        var station = _radio.currentStationName
        Gui.label(station != "" ? "Now: %(station)" : "--- static ---")
      } else {
        Gui.label("Radio not loaded")
      }
      
      if (_hitNodeName != "") {
        Gui.layoutRowDynamic(18, 1)
        Gui.label("Hit: %(_hitNodeName)")
      }
      
      Gui.layoutRowDynamic(18, 1)
      Gui.label("Right-click + drag: Look")
      Gui.label("WASD: Move camera")
      Gui.endWindow()
    }
  }

  quit() {}
}
