// Plume3D Wren script - Interactive Radio component
// Supports click-and-drag interactions for body rotation, volume, and frequency knobs

import "engine" for Audio, Config, Input, Logger, Math, AnimationPlayOptions, AnimationEvent

// Tuning constants
var TUNE_MIN = 88.0
var TUNE_MAX = 108.0
var TUNE_WIDTH = 2.5

// Drag sensitivity
var BODY_DRAG_SENSITIVITY = 0.005
var VOLUME_DRAG_SENSITIVITY = 0.008
var FREQUENCY_DRAG_SENSITIVITY = 0.15

class Radio {
    construct new() {
        _rootNode = null
        _config = null
        _nodes = null
        
        // Child nodes
        _powerButton = null
        _bodyNode = null
        _volumeKnob = null
        _frequencyKnob = null
        _frequencyTuner = null
        
        // Animation state
        _powerAnimState = null
        
        // Radio state
        _powerOn = false
        _frequency = 98.0
        _volume = 0.7
        _currentStationName = ""
        
        // Audio
        _channels = []
        _sources = []
        _staticSource = null
        
        // Drag state
        _isDragging = false
        _dragTarget = ""  // "body", "volume", "frequency"
        _dragStartMouseX = 0.0
        _dragStartMouseY = 0.0
        _dragStartBodyRotationX = 0.0
        _dragStartBodyRotationY = 0.0
        _dragStartVolume = 0.0
        _dragStartFrequency = 0.0
    }

    // Called by engine after instantiation
    init(rootNode, config) {
        _rootNode = rootNode
        _config = config
        _nodes = rootNode != null ? rootNode.nodes : null
        
        if (_nodes == null) {
            Logger.warn("Radio: no nodes map available")
            return
        }
        
        // Get child nodes by nodeId
        _powerButton = _nodes.containsKey("power_button") ? _nodes["power_button"] : null
        _bodyNode = _nodes.containsKey("radio_body") ? _nodes["radio_body"] : null
        _volumeKnob = _nodes.containsKey("knob_volume") ? _nodes["knob_volume"] : null
        _frequencyKnob = _nodes.containsKey("knob_frequency") ? _nodes["knob_frequency"] : null
        _frequencyTuner = _nodes.containsKey("frequency_tuner") ? _nodes["frequency_tuner"] : null
        
        // Set up power button animation
        if (_powerButton != null) {
            _powerAnimState = _powerButton.animations["Radio_Power_Button_Toggle"]
            if (_powerAnimState != null) {
                _powerAnimState.setFloat("power_on", 0)
                var events = _powerAnimState.events
                if (!events.containsKey("onStart")) events["onStart"] = AnimationEvent.new()
                if (!events.containsKey("onEnd")) events["onEnd"] = AnimationEvent.new()
            }
        }
        
        // Load audio channels from config
        var radioConfig = Config.load("Scripts/radio.toml")
        var channelList = radioConfig["channels"]
        if (channelList == null) channelList = []
        
        for (ch in channelList) {
            _channels.add(ch)
            var pathStr = ch["path"]
            var src = Audio.newSource(pathStr)
            if (src != null) {
                src.setLooping(true)
                _sources.add(src)
            } else {
                Logger.warn("Radio: failed to load audio '%(pathStr)'")
                _sources.add(null)
            }
        }
        
        _staticSource = Audio.newSource("sounds/static.wav")
        if (_staticSource != null) _staticSource.setLooping(true)
        
        Logger.info("Radio initialized: %(rootNode.name)")
        Logger.info("  Power button: %(_powerButton != null)")
        Logger.info("  Body node: %(_bodyNode != null)")
        Logger.info("  Volume knob: %(_volumeKnob != null)")
        Logger.info("  Frequency knob: %(_frequencyKnob != null)")
        Logger.info("  Channels loaded: %(_channels.count)")
    }

    // Accessors
    rootNode { _rootNode }
    config { _config }
    isDragging { _isDragging }
    powerOn { _powerOn }
    frequency { _frequency }
    volume { _volume }
    currentStationName { _currentStationName }
    
    // Check if a raycast hit belongs to this radio and handle interaction
    // Returns true if interaction was handled
    tryStartInteraction(hit, mouseX, mouseY) {
        if (hit == null || _rootNode == null) return false
        
        var hitNodeName = hit.getNodeName()
        
        // Power button click
        if (_powerButton != null && hitNodeName == _powerButton.getName()) {
            togglePower()
            return true
        }
        
        // Body drag (check for OBRadio_Body or the nodeId name)
        if (_bodyNode != null && (hitNodeName == "OBRadio_Body" || hitNodeName == _bodyNode.getName())) {
            startBodyDrag(mouseX, mouseY)
            return true
        }
        
        // Volume knob drag
        if (_volumeKnob != null && (hitNodeName == "OBRadio_Volume" || hitNodeName == _volumeKnob.getName())) {
            startVolumeDrag(mouseY)
            return true
        }
        
        // Frequency knob drag (either the knob or the tuner bar)
        if (_frequencyKnob != null && (hitNodeName == "OBRadio_Frequency" || hitNodeName == _frequencyKnob.getName())) {
            startFrequencyDrag(mouseX)
            return true
        }
        if (_frequencyTuner != null && (hitNodeName == "OBRadio_Freq_Tuner" || hitNodeName == _frequencyTuner.getName())) {
            startFrequencyDrag(mouseX)
            return true
        }
        
        return false
    }
    
    // Toggle power state
    togglePower() {
        if (_powerAnimState != null) {
            var opts = AnimationPlayOptions.new()
            opts.reverse = _powerOn
            _powerAnimState.play(opts)
            Logger.info("Radio: toggling power (was %(_powerOn))")
        } else {
            // No animation, just toggle directly
            _powerOn = !_powerOn
            Logger.info("Radio: power %(_powerOn ? "on" : "off")")
        }
    }
    
    // Start body rotation drag
    startBodyDrag(mouseX, mouseY) {
        if (_bodyNode == null) return
        
        _isDragging = true
        _dragTarget = "body"
        _dragStartMouseX = mouseX
        _dragStartMouseY = mouseY
        var rot = _bodyNode.getRotation()
        _dragStartBodyRotationX = rot[0]
        _dragStartBodyRotationY = rot[1]
        Logger.info("Radio: started body drag")
    }
    
    // Start volume drag
    startVolumeDrag(mouseY) {
        _isDragging = true
        _dragTarget = "volume"
        _dragStartMouseY = mouseY
        _dragStartVolume = _volume
        Logger.info("Radio: started volume drag")
    }
    
    // Start frequency drag
    startFrequencyDrag(mouseX) {
        _isDragging = true
        _dragTarget = "frequency"
        _dragStartMouseX = mouseX
        _dragStartFrequency = _frequency
        Logger.info("Radio: started frequency drag")
    }
    
    // Update drag interaction
    // Call this every frame while mouse is held
    updateDrag(mouseX, mouseY) {
        if (!_isDragging) return
        
        if (_dragTarget == "body" && _bodyNode != null) {
            var deltaX = mouseX - _dragStartMouseX
            var deltaY = mouseY - _dragStartMouseY
            var rot = _bodyNode.getRotation()
            // Horizontal drag rotates around Y, vertical drag rotates around X
            var newRotX = _dragStartBodyRotationX + deltaY * BODY_DRAG_SENSITIVITY
            var newRotY = _dragStartBodyRotationY + deltaX * BODY_DRAG_SENSITIVITY
            _bodyNode.setRotation(newRotX, newRotY, rot[2])
        } else if (_dragTarget == "volume") {
            var deltaY = _dragStartMouseY - mouseY  // Invert: drag up = increase
            _volume = Math.max(0, Math.min(1, _dragStartVolume + deltaY * VOLUME_DRAG_SENSITIVITY))
        } else if (_dragTarget == "frequency") {
            var deltaX = mouseX - _dragStartMouseX
            _frequency = Math.max(TUNE_MIN, Math.min(TUNE_MAX, _dragStartFrequency + deltaX * FREQUENCY_DRAG_SENSITIVITY))
        }
    }
    
    // End drag interaction
    endDrag() {
        if (_isDragging) {
            _isDragging = false
            _dragTarget = ""
        }
    }
    
    // Update radio state (call every frame)
    update(dt) {
        // Process animation events
        if (_powerAnimState != null) {
            for (e in _powerAnimState.pendingEvents) {
                if (_powerAnimState.events.containsKey(e.name)) {
                    _powerAnimState.events[e.name].invoke(e.name, e.time)
                }
            }
            var powerVal = _powerAnimState.getFloat("power_on")
            _powerOn = powerVal != null && powerVal > 0.5
        }
        
        // Find closest channel
        var closestIdx = -1
        var closestDist = 999.0
        var i = 0
        for (ch in _channels) {
            var dist = Math.abs(ch["frequency"] - _frequency)
            if (dist < closestDist) {
                closestDist = dist
                closestIdx = i
            }
            i = i + 1
        }
        
        // Calculate signal strength
        var signalStrength = 0.0
        if (closestIdx >= 0 && closestDist <= TUNE_WIDTH) {
            signalStrength = 1.0 - (closestDist / TUNE_WIDTH)
        }
        var staticGain = 1.0 - signalStrength
        
        // Update audio playback
        if (_powerOn) {
            for (i in 0..._sources.count) {
                var src = _sources[i]
                if (src != null) {
                    if (i == closestIdx && signalStrength > 0.01) {
                        src.setVolume(signalStrength * _volume)
                        if (!Audio.isPlaying(src)) Audio.play(src)
                    } else {
                        Audio.stop(src)
                    }
                }
            }
            if (_staticSource != null) {
                if (staticGain > 0.01) {
                    _staticSource.setVolume(staticGain * _volume)
                    if (!Audio.isPlaying(_staticSource)) Audio.play(_staticSource)
                } else {
                    Audio.stop(_staticSource)
                }
            }
        } else {
            // Power off - stop all audio
            for (src in _sources) {
                if (src != null) Audio.stop(src)
            }
            if (_staticSource != null) Audio.stop(_staticSource)
        }
        
        // Update current station name
        _currentStationName = ""
        if (closestIdx >= 0 && signalStrength > 0.5) {
            _currentStationName = _channels[closestIdx]["name"]
        }
    }
    
    // Set volume directly (0-1)
    setVolume(v) {
        _volume = Math.max(0, Math.min(1, v))
    }
    
    // Set frequency directly (TUNE_MIN to TUNE_MAX)
    setFrequency(f) {
        _frequency = Math.max(TUNE_MIN, Math.min(TUNE_MAX, f))
    }
    
    // Get body rotation
    getBodyRotation() {
        if (_bodyNode != null) {
            return _bodyNode.getRotation()
        }
        return [0, 0, 0]
    }
    
    // Set body rotation
    setBodyRotation(x, y, z) {
        if (_bodyNode != null) {
            _bodyNode.setRotation(x, y, z)
        }
    }
}
