# SPDX-License-Identifier: MIT
# Plume3D Wren Script Config addon: attach Wren script config to objects/collections.
# Config and scripts live in the project (source control); .blend stores only references (config path, script path, class name).
# Engine loads .blend; validates config path and script path and that class exists in script; runtime exception if not found.

bl_info = {
    "name": "Plume3D",
    "author": "Plume3D",
    "version": (0, 0, 1),
    "blender_version": (5, 0, 0),
    "location": "Viewport Sidebar (N) > Plume3D",
    "description": "Attach Wren script/config to objects and collections from the viewport. Config and scripts live in the project; .blend stores references only.",
    "category": "Development",
}

import os

import bpy
from bpy.props import (
    StringProperty,
    BoolProperty,
    FloatProperty,
    IntProperty,
    EnumProperty,
    CollectionProperty,
    PointerProperty,
)
from bpy.types import PropertyGroup, Operator, Panel, UIList, AddonPreferences


# --- Layer names: project-specific; use defaults when not loaded from game.toml ---
DEFAULT_TAG_LAYER_NAMES = "Default,Interactable"
DEFAULT_COLLISION_LAYER_NAMES = "Default,Interactable"



def _parse_layer_names(s):
    """Parse comma-separated layer names; strip whitespace; skip empty."""
    if not s:
        return []
    return [x.strip() for x in s.split(",") if x.strip()]


class PLUME3D_AddonPreferences(AddonPreferences):
    bl_idname = __name__


# --- Custom properties on ID (Object or Collection): reference only (paths + class name) ---
class PLUME3D_WrenScriptConfig(PropertyGroup):
    """Stored on Object or Collection. Only references: config path, script path, class name. Files live in project."""
    enabled: BoolProperty(name="Enable Wren Script Config", default=False)
    # Path to TOML file in the project (e.g. "radio.toml"). Loaded from project at runtime; not embedded in .blend.
    config_path: StringProperty(
        name="Config Path",
        description="Path to TOML file in project (e.g. radio.toml). Engine loads it when loading the .blend; validation fails with runtime exception if not found.",
        default="",
    )
    # Path to Wren script in the project (e.g. "main.wren" or "scripts/radio.wren"). Loaded from project at runtime; not embedded in .blend.
    script_path: StringProperty(
        name="Script Path",
        description="Path to Wren script in project (e.g. main.wren). A script can have multiple classes; Class Name selects which one to use for this object/collection. Validation fails with runtime exception if file or class not found.",
        default="",
    )
    # Which class in the script to use (e.g. Radio). Engine validates that this class exists in the script at load; runtime exception if not found.
    class_name: StringProperty(
        name="Class Name",
        description="Class in the script to hook into for this object/collection (e.g. Radio). Required because a Wren script can define more than one class.",
        default="",
    )
    # Stable id for this node within an instantiated hierarchy; used as .nodes[node_id] on the root instance.
    node_id: StringProperty(
        name="Node Id",
        description="Stable identifier for this node within an instantiated hierarchy. The root instance exposes .nodes[node_id] to access this node.",
        default="",
    )
    # Bitmask for tags (raycast, gameplay, filtering). Engine interprets as bit flags.
    tags: IntProperty(
        name="Tags",
        description="Bitmasked tags for this object/collection (e.g. for raycast or gameplay filtering).",
        default=0,
        min=0,
    )
    # Bitmask for collision layers (which layers this object is on for raycast/collision).
    collision_layers: IntProperty(
        name="Collision Layers",
        description="Bitmasked collision layers for raycast and collision (which layers this object is on).",
        default=0,
        min=0,
    )
    # Game config path (e.g. game.toml) for root collection; drives tag/collision layer names for the file.
    game_config_path: StringProperty(
        name="Game Config Path",
        description="Path to game.toml (or game config) for this blend file. Set on the root collection to drive tag and collision layer names for every node.",
        default="",
    )
    # Per-object shader name (Slang module key). Resolution: material > object > default.
    shader_name: StringProperty(
        name="Shader Name",
        description="Slang shader module key for this object (e.g. triangle). Resolution order: material shader > object shader > default.",
        default="",
    )


# --- File browser / create-new operators: project root = directory containing game.toml ---
def _find_project_root(start_path):
    """Walk up from start_path (file or dir) until we find a directory containing game.toml; return that directory or None."""
    if not start_path or not start_path.strip():
        return None
    current = os.path.normpath(os.path.abspath(start_path))
    if not os.path.isdir(current):
        current = os.path.dirname(current)
    while current:
        game_toml = os.path.join(current, "game.toml")
        if os.path.isfile(game_toml):
            return current
        parent = os.path.dirname(current)
        if parent == current:
            break
        current = parent
    return None


def _get_project_root_for_browse(context):
    """Return project root for opening the file browser: from blend file dir if saved, else None."""
    if not bpy.data.filepath:
        return None
    return _find_project_root(os.path.dirname(bpy.data.filepath))


def _path_relative_to_root(path, root):
    """Return path relative to root; if root is None or path not under root, return path unchanged."""
    if not path or not root:
        return path
    try:
        return os.path.relpath(os.path.normpath(os.path.abspath(path)), os.path.normpath(os.path.abspath(root)))
    except (ValueError, OSError):
        return path


def _path_inside_root(path, root):
    """True if path is inside root (both normalized/real)."""
    if not path or not root:
        return False
    try:
        real_root = os.path.realpath(os.path.normpath(root))
        real_path = os.path.realpath(os.path.normpath(os.path.abspath(path)))
        return real_path == real_root or real_path.startswith(real_root + os.sep)
    except (ValueError, OSError):
        return False


def _set_initial_browse_path(context, operator, default_filename):
    """Set operator.filepath to project_root/default_filename so file browser opens in project. Project root from blend file dir."""
    root = _get_project_root_for_browse(context)
    if not root:
        return
    current = (getattr(operator, "filepath", None) or "").strip()
    if current and not os.path.isabs(current):
        current = os.path.normpath(os.path.join(root, current))
    if not current or not _path_inside_root(current, root):
        operator.filepath = os.path.join(root, default_filename)


def _get_config(context, target_object=None, target_collection=None):
    """Get plume3d_wren_config from optional target, or from context."""
    if target_object is not None:
        return target_object.plume3d_wren_config
    if target_collection is not None:
        return target_collection.plume3d_wren_config
    if context.object:
        return context.object.plume3d_wren_config
    if context.collection:
        return context.collection.plume3d_wren_config
    return None


def _cfg_owner_name(cfg, obj, coll):
    """Return a string identifying which ID owns cfg (for logging). _get_config prefers obj over coll."""
    if obj:
        return "Object '%s'" % obj.name
    if coll:
        return "Collection '%s'" % coll.name
    return "unknown"


class PLUME3D_OT_BrowseConfigPath(Operator):
    bl_idname = "plume3d.browse_config_path"
    bl_label = "Browse for Config"
    bl_options = {"REGISTER", "INTERNAL"}
    filepath: StringProperty(subtype="FILE_PATH", options={"HIDDEN", "SKIP_SAVE"})
    filter_glob: StringProperty(default="*.toml", options={"HIDDEN"})

    def invoke(self, context, event):
        obj, coll, err = _get_single_selection_for_config(context)
        if err:
            self.report({"WARNING"}, err)
            return {"CANCELLED"}
        cfg = _get_config(context, target_object=obj, target_collection=coll)
        if cfg and cfg.config_path:
            self.filepath = cfg.config_path
        _set_initial_browse_path(context, self, "config.toml")
        context.window_manager.fileselect_add(self)
        return {"RUNNING_MODAL"}

    def execute(self, context):
        obj, coll, err = _get_single_selection_for_config(context)
        if err:
            self.report({"WARNING"}, err)
            return {"CANCELLED"}
        cfg = _get_config(context, target_object=obj, target_collection=coll)
        if cfg is None:
            return {"CANCELLED"}
        path = os.path.normpath(os.path.abspath(self.filepath))
        project_root = _find_project_root(os.path.dirname(path))
        if project_root is None:
            self.report({"ERROR"}, "No game.toml found in the directory tree. Select a file inside a Plume3D project.")
            return {"CANCELLED"}
        if not _path_inside_root(path, project_root):
            self.report({"ERROR"}, "Path must be inside the Plume3D project directory (above game.toml).")
            return {"CANCELLED"}
        value = _path_relative_to_root(path, project_root) or self.filepath
        cfg.config_path = value
        self.report({"INFO"}, "Config path set to %s on %s" % (value, _cfg_owner_name(cfg, obj, coll)))
        return {"FINISHED"}


class PLUME3D_OT_BrowseGameConfigPath(Operator):
    bl_idname = "plume3d.browse_game_config_path"
    bl_label = "Browse for Game Config"
    bl_options = {"REGISTER", "INTERNAL"}
    filepath: StringProperty(subtype="FILE_PATH", options={"HIDDEN", "SKIP_SAVE"})
    filter_glob: StringProperty(default="*.toml", options={"HIDDEN"})

    def invoke(self, context, event):
        obj, coll, err = _get_single_selection_for_config(context)
        if err:
            self.report({"WARNING"}, err)
            return {"CANCELLED"}
        cfg = _get_config(context, target_object=obj, target_collection=coll)
        if cfg and cfg.game_config_path:
            self.filepath = cfg.game_config_path
        _set_initial_browse_path(context, self, "game.toml")
        context.window_manager.fileselect_add(self)
        return {"RUNNING_MODAL"}

    def execute(self, context):
        obj, coll, err = _get_single_selection_for_config(context)
        if err:
            self.report({"WARNING"}, err)
            return {"CANCELLED"}
        cfg = _get_config(context, target_object=obj, target_collection=coll)
        if cfg is None:
            return {"CANCELLED"}
        path = getattr(self, "filepath", None) or ""
        if path:
            path = os.path.normpath(os.path.abspath(path))
            project_root = _find_project_root(os.path.dirname(path))
            if project_root is None:
                self.report({"ERROR"}, "No game.toml found in the directory tree. Select a file inside a Plume3D project.")
                return {"CANCELLED"}
            if not _path_inside_root(path, project_root):
                self.report({"ERROR"}, "Path must be inside the Plume3D project directory (above game.toml).")
                return {"CANCELLED"}
            value = _path_relative_to_root(path, project_root) or path
            cfg.game_config_path = value
            self.report({"INFO"}, "Game config path set to %s on %s" % (value, _cfg_owner_name(cfg, obj, coll)))
        return {"FINISHED"}


class PLUME3D_OT_CreateNewConfig(Operator):
    bl_idname = "plume3d.create_new_config"
    bl_label = "Create New Config"
    bl_options = {"REGISTER", "INTERNAL"}
    filepath: StringProperty(subtype="FILE_PATH", options={"HIDDEN", "SKIP_SAVE"})
    filter_glob: StringProperty(default="*.toml", options={"HIDDEN"})

    def invoke(self, context, event):
        obj, coll, err = _get_single_selection_for_config(context)
        if err:
            self.report({"WARNING"}, err)
            return {"CANCELLED"}
        cfg = _get_config(context, target_object=obj, target_collection=coll)
        if cfg and cfg.config_path:
            self.filepath = cfg.config_path
        _set_initial_browse_path(context, self, "config.toml")
        context.window_manager.fileselect_add(self)
        return {"RUNNING_MODAL"}

    def execute(self, context):
        obj, coll, err = _get_single_selection_for_config(context)
        if err:
            self.report({"WARNING"}, err)
            return {"CANCELLED"}
        cfg = _get_config(context, target_object=obj, target_collection=coll)
        if cfg is None:
            return {"CANCELLED"}
        path = self.filepath
        if path and not path.endswith(".toml"):
            path = path + ".toml"
        path = os.path.normpath(os.path.abspath(path))
        project_root = _find_project_root(os.path.dirname(path))
        if project_root is None:
            self.report({"ERROR"}, "No game.toml found in the directory tree. Select a file inside a Plume3D project.")
            return {"CANCELLED"}
        if not _path_inside_root(path, project_root):
            self.report({"ERROR"}, "Path must be inside the Plume3D project directory (above game.toml).")
            return {"CANCELLED"}
        try:
            with open(path, "w", encoding="utf-8") as f:
                f.write("# Plume3D config\n\n")
        except OSError:
            pass
        value = _path_relative_to_root(path, project_root) or path
        cfg.config_path = value
        self.report({"INFO"}, "Config path set to %s on %s" % (value, _cfg_owner_name(cfg, obj, coll)))
        return {"FINISHED"}


class PLUME3D_OT_BrowseScriptPath(Operator):
    bl_idname = "plume3d.browse_script_path"
    bl_label = "Browse for Script"
    bl_options = {"REGISTER", "INTERNAL"}
    filepath: StringProperty(subtype="FILE_PATH", options={"HIDDEN", "SKIP_SAVE"})
    filter_glob: StringProperty(default="*.wren", options={"HIDDEN"})

    def invoke(self, context, event):
        obj, coll, err = _get_single_selection_for_config(context)
        if err:
            self.report({"WARNING"}, err)
            return {"CANCELLED"}
        cfg = _get_config(context, target_object=obj, target_collection=coll)
        if cfg and cfg.script_path:
            self.filepath = cfg.script_path
        _set_initial_browse_path(context, self, "script.wren")
        context.window_manager.fileselect_add(self)
        return {"RUNNING_MODAL"}

    def execute(self, context):
        obj, coll, err = _get_single_selection_for_config(context)
        if err:
            self.report({"WARNING"}, err)
            return {"CANCELLED"}
        cfg = _get_config(context, target_object=obj, target_collection=coll)
        if cfg is None:
            return {"CANCELLED"}
        path = os.path.normpath(os.path.abspath(self.filepath))
        project_root = _find_project_root(os.path.dirname(path))
        if project_root is None:
            self.report({"ERROR"}, "No game.toml found in the directory tree. Select a file inside a Plume3D project.")
            return {"CANCELLED"}
        if not _path_inside_root(path, project_root):
            self.report({"ERROR"}, "Path must be inside the Plume3D project directory (above game.toml).")
            return {"CANCELLED"}
        value = _path_relative_to_root(path, project_root) or self.filepath
        cfg.script_path = value
        self.report({"INFO"}, "Class path set to %s on %s" % (value, _cfg_owner_name(cfg, obj, coll)))
        return {"FINISHED"}


class PLUME3D_OT_CreateNewScript(Operator):
    bl_idname = "plume3d.create_new_script"
    bl_label = "Create New Script"
    bl_options = {"REGISTER", "INTERNAL"}
    filepath: StringProperty(subtype="FILE_PATH", options={"HIDDEN", "SKIP_SAVE"})
    filter_glob: StringProperty(default="*.wren", options={"HIDDEN"})

    def invoke(self, context, event):
        obj, coll, err = _get_single_selection_for_config(context)
        if err:
            self.report({"WARNING"}, err)
            return {"CANCELLED"}
        cfg = _get_config(context, target_object=obj, target_collection=coll)
        if cfg and cfg.script_path:
            self.filepath = cfg.script_path
        _set_initial_browse_path(context, self, "script.wren")
        context.window_manager.fileselect_add(self)
        return {"RUNNING_MODAL"}

    def execute(self, context):
        obj, coll, err = _get_single_selection_for_config(context)
        if err:
            self.report({"WARNING"}, err)
            return {"CANCELLED"}
        cfg = _get_config(context, target_object=obj, target_collection=coll)
        if cfg is None:
            return {"CANCELLED"}
        path = self.filepath
        if path and not path.endswith(".wren"):
            path = path + ".wren"
        path = os.path.normpath(os.path.abspath(path))
        project_root = _find_project_root(os.path.dirname(path))
        if project_root is None:
            self.report({"ERROR"}, "No game.toml found in the directory tree. Select a file inside a Plume3D project.")
            return {"CANCELLED"}
        if not _path_inside_root(path, project_root):
            self.report({"ERROR"}, "Path must be inside the Plume3D project directory (above game.toml).")
            return {"CANCELLED"}
        try:
            with open(path, "w", encoding="utf-8") as f:
                f.write("// Plume3D Wren script\n\n")
        except OSError:
            pass
        value = _path_relative_to_root(path, project_root) or path
        cfg.script_path = value
        self.report({"INFO"}, "Class path set to %s on %s" % (value, _cfg_owner_name(cfg, obj, coll)))
        return {"FINISHED"}


# --- Operators ---
class PLUME3D_OT_ToggleTagLayer(Operator):
    """Toggle one tag layer (bit) for the selected object/collection."""
    bl_idname = "plume3d.toggle_tag_layer"
    bl_label = "Toggle Tag Layer"
    bl_options = {"REGISTER", "UNDO"}
    layer_index: IntProperty(default=0)

    def execute(self, context):
        obj, coll, err = _get_single_selection_for_config(context)
        obj_or_coll = (obj or coll) if not err else (context.object or context.collection)
        if obj_or_coll is None:
            return {"CANCELLED"}
        cfg = obj_or_coll.plume3d_wren_config
        i = self.layer_index
        if i < 0 or i > 31:
            return {"CANCELLED"}
        bit = 1 << i
        cfg.tags = cfg.tags ^ bit
        return {"FINISHED"}


class PLUME3D_OT_ToggleCollisionLayer(Operator):
    """Toggle one collision layer (bit) for the selected object/collection."""
    bl_idname = "plume3d.toggle_collision_layer"
    bl_label = "Toggle Collision Layer"
    bl_options = {"REGISTER", "UNDO"}
    layer_index: IntProperty(default=0)

    def execute(self, context):
        obj, coll, err = _get_single_selection_for_config(context)
        obj_or_coll = (obj or coll) if not err else (context.object or context.collection)
        if obj_or_coll is None:
            return {"CANCELLED"}
        cfg = obj_or_coll.plume3d_wren_config
        i = self.layer_index
        if i < 0 or i > 31:
            return {"CANCELLED"}
        bit = 1 << i
        cfg.collision_layers = cfg.collision_layers ^ bit
        return {"FINISHED"}


# --- Root-only: Paths (Class Name, Config Path, Class Path) are only for roots; children inherit from an ancestor ---
def _parent_collection_of(coll):
    """Return the collection that contains coll as a direct child, or None."""
    if not coll:
        return None
    for scene in bpy.data.scenes:
        def find_parent(c, target):
            if not c:
                return None
            for ch in c.children:
                if ch == target:
                    return c
                p = find_parent(ch, target)
                if p:
                    return p
            return None
        p = find_parent(scene.collection, coll)
        if p:
            return p
    return None


def _config_has_class_root(cfg):
    """True if this config is enabled and has class_name, config_path, or script_path set (i.e. a class root)."""
    if not cfg or not getattr(cfg, "enabled", False):
        return False
    return bool(
        getattr(cfg, "class_name", "") or getattr(cfg, "config_path", "") or getattr(cfg, "script_path", "")
    )


def _collection_or_ancestor_has_class_root(coll):
    """True if this collection or any ancestor has class (class_name/config_path/script_path) set."""
    while coll:
        cfg = getattr(coll, "plume3d_wren_config", None)
        if _config_has_class_root(cfg):
            return True
        coll = _parent_collection_of(coll)
    return False


def _any_ancestor_has_class_configured(target_object=None, target_collection=None):
    """True if any ancestor (object parent chain or collection hierarchy) has Class Name / Config Path / Class Path set.
    When true, this node is a child of a class root and Paths should be hidden (only Node Id and layers shown)."""
    if target_object is not None:
        # Walk object parent chain
        obj = getattr(target_object, "parent", None)
        while obj and isinstance(obj, bpy.types.Object):
            cfg = getattr(obj, "plume3d_wren_config", None)
            if _config_has_class_root(cfg):
                return True
            obj = getattr(obj, "parent", None)
        # Check every collection this object is in (and their ancestors)
        users_collection = getattr(target_object, "users_collection", None)
        if users_collection:
            for coll in users_collection:
                if _collection_or_ancestor_has_class_root(coll):
                    return True
        return False
    if target_collection is not None:
        return _collection_or_ancestor_has_class_root(_parent_collection_of(target_collection))
    return False


def _any_ancestor_has_game_config(target_object=None, target_collection=None):
    """True if this collection/object or any ancestor collection has game_config_path set.
    Used to show Tags & Collision and to hide 'no game config' message for descendants."""
    return _get_ancestor_game_config_path(target_object, target_collection) is not None


def _get_ancestor_game_config_path(target_object=None, target_collection=None):
    """Return the game_config_path string from this or an ancestor collection, or None."""
    if target_collection is not None:
        coll = target_collection
        while coll:
            cfg = getattr(coll, "plume3d_wren_config", None)
            path = cfg and getattr(cfg, "game_config_path", "").strip()
            if path:
                return path
            coll = _parent_collection_of(coll)
        return None
    if target_object is not None:
        users_collection = getattr(target_object, "users_collection", None)
        if users_collection:
            for coll in users_collection:
                p = _get_ancestor_game_config_path(target_collection=coll)
                if p:
                    return p
        return None
    return None


def _parse_physics_layer_names_from_toml(filepath):
    """Parse [Physics].tag_layers and [Physics].collision_layers from a game.toml file.
    Returns (tag_names list, collision_names list, status).
    status: "error" (file/parse failed), "not_configured" ([Physics] found but both lists empty),
    "ok" (at least one list non-empty)."""
    if not filepath or not os.path.isfile(filepath):
        return (None, None, "error")
    try:
        with open(filepath, "rb") as f:
            data = f.read().decode("utf-8", errors="replace")
    except OSError:
        return (None, None, "error")
    # Minimal parse: find [Physics] then tag_layers = [...] and collision_layers = [...]
    tag_names = None
    collision_names = None
    in_physics = False
    for line in data.splitlines():
        line = line.strip()
        if line == "[Physics]":
            in_physics = True
            continue
        if line.startswith("["):
            in_physics = False
            continue
        if not in_physics:
            continue
        if line.startswith("tag_layers"):
            tag_names = _parse_toml_array_line(line)
            if tag_names is not None:
                continue
        if line.startswith("collision_layers"):
            collision_names = _parse_toml_array_line(line)
    if not in_physics:
        # File exists but [Physics] section was never found
        return (None, None, "error")
    if tag_names is None:
        tag_names = []
    if collision_names is None:
        collision_names = []
    if not tag_names and not collision_names:
        return (tag_names, collision_names, "not_configured")
    return (tag_names, collision_names, "ok")


def _parse_toml_array_line(line):
    """Parse a TOML line like tag_layers = [] or collision_layers = ["A", "B"] into a list of strings."""
    eq = line.find("=")
    if eq < 0:
        return None
    rest = line[eq + 1:].strip()
    if not rest.startswith("[") or not rest.endswith("]"):
        return None
    inner = rest[1:-1].strip()
    if not inner:
        return []
    result = []
    for part in inner.split(","):
        part = part.strip()
        if (part.startswith('"') and part.endswith('"')) or (part.startswith("'") and part.endswith("'")):
            result.append(part[1:-1].replace('\\"', '"').replace("\\'", "'"))
        elif part:
            result.append(part)
    return result


# --- Hierarchy selection: resolve selected object or collection from Outliner ---
def _get_selected_ids_from_outliner(context):
    """Return list of selected IDs (Objects/Collections) from Outliner, or None if no Outliner."""
    if not hasattr(context, "temp_override"):
        return None
    for area in context.screen.areas:
        if area.type != "OUTLINER":
            continue
        try:
            override = {"area": area}
            for region in area.regions:
                if region.type == "WINDOW":
                    override["region"] = region
                    break
            with context.temp_override(**override):
                ids = getattr(context, "selected_ids", None)
                if ids is not None:
                    return list(ids)
        except Exception:
            try:
                with context.temp_override(area=area):
                    ids = getattr(context, "selected_ids", None)
                    if ids is not None:
                        return list(ids)
            except Exception:
                pass
    return None


def _get_single_selection_for_config(context):
    """Return (obj, coll, err_msg). Exactly one of obj/coll set and err_msg None, or both None and err_msg set.
    Use this for path operators: target is always 'current selection'; multi-select is not allowed."""
    ids = _get_selected_ids_from_outliner(context)
    if ids is not None:
        # Filter to objects and collections only
        objs = [x for x in ids if isinstance(x, bpy.types.Object)]
        colls = [x for x in ids if isinstance(x, bpy.types.Collection)]
        n = len(objs) + len(colls)
        if n == 0:
            return (None, None, "Select an object or collection in the Outliner.")
        if n > 1:
            return (None, None, "Multi-select editing is currently not available.")
        if objs:
            return (objs[0], None, None)
        return (None, colls[0], None)
    # No Outliner: fallback to context.object only (single selection)
    obj = context.object
    if obj:
        return (obj, None, None)
    return (None, None, "Select an object or collection in the Outliner.")


def _get_selection_from_hierarchy(context):
    """Return ("object", obj) or ("collection", coll) or (None, None) from Outliner selection.
    Prefer object if both are selected. Falls back to context.object when no Outliner area."""
    obj, coll, err = _get_single_selection_for_config(context)
    if err:
        return (None, None)
    if obj:
        return ("object", obj)
    return ("collection", coll)


def _draw_named_layers(context, layout, cfg, target_object, target_collection, prefs):
    """Draw Tags and Collision Layers. Layer names come from game.toml [Physics] when game_config_path is set, else addon preferences."""
    def op_target(op):
        # Operators resolve target from context (Blender 5+ Operators don't support PointerProperty).
        pass

    # Prefer layer names from the project's game.toml when an ancestor has game_config_path set
    tag_names = None
    coll_names = None
    status = None  # "error", "not_configured", "ok", or None when game.toml not used
    rel_path = _get_ancestor_game_config_path(target_object, target_collection)
    if rel_path and context and getattr(bpy.data, "filepath", None):
        project_root = _find_project_root(os.path.dirname(bpy.data.filepath))
        if project_root:
            abs_path = os.path.normpath(os.path.join(project_root, rel_path))
            tag_names, coll_names, status = _parse_physics_layer_names_from_toml(abs_path)
    # Show message and fall back to prefs for error / not_configured
    if status == "error":
        layout.label(text="There was an error reading the game.toml physics section.", icon="ERROR")
        tag_names = None
        coll_names = None
    elif status == "not_configured":
        layout.label(text="Tags and collision layers are not configured in game.toml; collisions are not available.", icon="INFO")
        tag_names = None
        coll_names = None
    if tag_names is None:
        tag_names = _parse_layer_names((getattr(prefs, "tag_layer_names", None) or DEFAULT_TAG_LAYER_NAMES) if prefs else DEFAULT_TAG_LAYER_NAMES)
    if tag_names:
        layout.label(text="Tags (named layers):")
        row = layout.row(align=True)
        for i, name in enumerate(tag_names):
            if i > 0 and i % 4 == 0:
                row = layout.row(align=True)
            bit = 1 << i
            on = (cfg.tags & bit) != 0
            op = row.operator("plume3d.toggle_tag_layer", text=name, icon="CHECKBOX_HLT" if on else "CHECKBOX_DEHLT", depress=on)
            op.layer_index = i
            op_target(op)
    else:
        if status == "ok" and rel_path:
            layout.label(text="Tags: not configured in game.toml")
        else:
            layout.prop(cfg, "tags", text="Tags (bitmask)")

    if coll_names is None:
        coll_names = _parse_layer_names((getattr(prefs, "collision_layer_names", None) or DEFAULT_COLLISION_LAYER_NAMES) if prefs else DEFAULT_COLLISION_LAYER_NAMES)
    if coll_names:
        layout.label(text="Collision layers (named):")
        row = layout.row(align=True)
        for i, name in enumerate(coll_names):
            if i > 0 and i % 4 == 0:
                row = layout.row(align=True)
            bit = 1 << i
            on = (cfg.collision_layers & bit) != 0
            op = row.operator("plume3d.toggle_collision_layer", text=name, icon="CHECKBOX_HLT" if on else "CHECKBOX_DEHLT", depress=on)
            op.layer_index = i
            op_target(op)
    else:
        if status == "ok" and rel_path:
            layout.label(text="Collision layers: not configured in game.toml")
        else:
            layout.prop(cfg, "collision_layers", text="Collision Layers (bitmask)")


# --- Shared draw: config UI (paths + legacy props) for a given cfg and optional target ---
def _draw_config_ui(context, layout, cfg, target_object=None, target_collection=None):
    """Draw Config Path, Class Path, Class Name, and legacy props. Pass target_* for hierarchy/ancestor logic only."""
    layout.prop(cfg, "enabled")
    if not cfg.enabled:
        return
    def op_target(op):
        # Operators resolve target from context (Blender 5+ Operators don't support PointerProperty).
        pass

    prefs = context.preferences.addons.get(__name__).preferences if context.preferences.addons.get(__name__) else None

    def _is_root_collection(coll, config):
        """True if collection is a root (direct child of scene, or has class_name as script root)."""
        if coll is None:
            return False
        if getattr(coll, "parent", None) == context.scene.collection:
            return True
        return bool(config and getattr(config, "class_name", None))

    is_child_of_class = _any_ancestor_has_class_configured(target_object, target_collection)
    has_game_config = _any_ancestor_has_game_config(target_object, target_collection)
    if is_child_of_class:
        layout.label(text="Paths inherited from parent — only Node Id, Shader and layers below", icon="INFO")
        box = layout.box()
        box.prop(cfg, "node_id", text="Node Id")
        box.prop(cfg, "shader_name", text="Shader Name")
        if has_game_config:
            _draw_named_layers(context, box, cfg, target_object, target_collection, prefs)
        else:
            box.label(text="No game config selected. Nodes in this collection will not be considered for tagging or collision.", icon="INFO")
        return

    box = layout.box()
    box.label(text="Paths (reference only)", icon="SCRIPT")
    # 1. Class Name (Wren class in the script)
    box.prop(cfg, "class_name", text="Class Name")
    # Game Config Path: only for root collections, directly under Class Name (same section)
    if target_collection is not None and _is_root_collection(target_collection, cfg):
        row = box.row(align=True)
        row.prop(cfg, "game_config_path", text="Game Config Path")
        op = row.operator("plume3d.browse_game_config_path", text="", icon="FILE_FOLDER")
        op_target(op)
    # 2. Config Path = path to .toml file (show before Class Path so it's visible without scrolling)
    row = box.row(align=True)
    row.prop(cfg, "config_path", text="Config Path")
    op = row.operator("plume3d.browse_config_path", text="", icon="FILE_FOLDER")
    op_target(op)
    op = row.operator("plume3d.create_new_config", text="", icon="FILE_NEW")
    op_target(op)
    # 3. Class Path = path to .wren script file
    row = box.row(align=True)
    row.prop(cfg, "script_path", text="Class Path")
    op = row.operator("plume3d.browse_script_path", text="", icon="FILE_FOLDER")
    op_target(op)
    op = row.operator("plume3d.create_new_script", text="", icon="FILE_NEW")
    op_target(op)
    if cfg.config_path:
        box.label(text="Config: %s (in project)" % cfg.config_path, icon="FILE")
    if cfg.script_path:
        box.label(text="Class path: %s — class: %s (in project)" % (cfg.script_path, cfg.class_name or "?"), icon="TEXT")
    box = layout.box()
    box.label(text="Node & layers", icon="OUTLINER")
    box.prop(cfg, "node_id", text="Node Id")
    box.prop(cfg, "shader_name", text="Shader Name")
    if has_game_config:
        _draw_named_layers(context, box, cfg, target_object, target_collection, prefs)
    else:
        box.label(text="No game config selected. Nodes in this collection will not be considered for tagging or collision.", icon="INFO")
# --- Viewport sidebar panel (primary UX: N-panel > Plume3D) ---
# Context-sensitive: show config for the object or collection selected in the Hierarchy (Outliner).
class PLUME3D_PT_WrenScriptConfigViewport(Panel):
    bl_label = "Wren Script Config"
    bl_idname = "PLUME3D_PT_wren_config_viewport"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = "Plume3D"

    @classmethod
    def poll(cls, context):
        return True

    def draw(self, context):
        layout = self.layout
        wm = context.window_manager
        obj, coll, err = _get_single_selection_for_config(context)
        if err:
            layout.label(text=err, icon="INFO")
            if "Multi-select" in err:
                layout.separator()
                layout.label(text="Select exactly one object or collection to edit paths.", icon="OUTLINER")
            else:
                layout.separator()
                layout.label(text="Or pick a collection:", icon="OUTLINER_COLLECTION")
                layout.prop(wm, "plume3d_edit_collection", text="")
                coll_edit = getattr(wm, "plume3d_edit_collection", None)
                if coll_edit:
                    box = layout.box()
                    box.label(text="Collection: %s" % coll_edit.name, icon="OUTLINER_COLLECTION")
                    if hasattr(context, "temp_override"):
                        with context.temp_override(collection=coll_edit):
                            _draw_config_ui(context, box, coll_edit.plume3d_wren_config, target_object=None, target_collection=coll_edit)
                    else:
                        _draw_config_ui(context, box, coll_edit.plume3d_wren_config, target_object=None, target_collection=coll_edit)
            return
        target = obj or coll
        sel_type = "object" if obj else "collection"
        if sel_type == "object" and target:
            box = layout.box()
            box.label(text="Object: %s" % target.name, icon="OBJECT_DATA")
            _draw_config_ui(context, box, target.plume3d_wren_config, target_object=target, target_collection=None)
        else:
            # sel_type == "collection"
            box = layout.box()
            box.label(text="Collection: %s" % target.name, icon="OUTLINER_COLLECTION")
            if hasattr(context, "temp_override"):
                with context.temp_override(collection=target):
                    _draw_config_ui(context, box, target.plume3d_wren_config, target_object=None, target_collection=target)
            else:
                _draw_config_ui(context, box, target.plume3d_wren_config, target_object=None, target_collection=target)


# --- Sync PropertyGroup to ID properties so C++ blend loader can read them ---
# The engine's blend loader reads ID properties (Custom Properties), not RNA
# PointerProperty. We sync plume3d_wren_config to flat id["plume3d_*"] on save
# so that LoadBlendFromMemory finds class names, node ids, and paths.
# Handlers: save_pre(dummy), load_post(dummy, context) — accept both so no manual sync is needed.
def _sync_plume3d_to_id_properties(dummy=None, context=None):
    """Write plume3d_wren_config from every Object/Collection to ID custom properties."""
    for obj in getattr(bpy.data, "objects", []):
        cfg = getattr(obj, "plume3d_wren_config", None)
        if cfg is None:
            continue
        _write_config_to_id(obj, cfg)
    for coll in getattr(bpy.data, "collections", []):
        cfg = getattr(coll, "plume3d_wren_config", None)
        if cfg is None:
            continue
        _write_config_to_id(coll, cfg)


def _write_config_to_id(id_block, cfg):
    """Write one config to id_block's custom properties (overwrites plume3d_*)."""
    id_block["plume3d_enabled"] = 1 if getattr(cfg, "enabled", False) else 0
    id_block["plume3d_class_name"] = getattr(cfg, "class_name", "") or ""
    id_block["plume3d_config_path"] = getattr(cfg, "config_path", "") or ""
    id_block["plume3d_script_path"] = getattr(cfg, "script_path", "") or ""
    id_block["plume3d_node_id"] = getattr(cfg, "node_id", "") or ""
    id_block["plume3d_tags"] = getattr(cfg, "tags", 0)
    id_block["plume3d_collision_layers"] = getattr(cfg, "collision_layers", 0)
    id_block["plume3d_shader_name"] = getattr(cfg, "shader_name", "") or ""
    id_block["plume3d_game_config_path"] = getattr(cfg, "game_config_path", "") or ""


# --- Register ---
def register():
    bpy.utils.register_class(PLUME3D_AddonPreferences)
    bpy.utils.register_class(PLUME3D_WrenScriptConfig)
    bpy.utils.register_class(PLUME3D_OT_BrowseConfigPath)
    bpy.utils.register_class(PLUME3D_OT_BrowseGameConfigPath)
    bpy.utils.register_class(PLUME3D_OT_CreateNewConfig)
    bpy.utils.register_class(PLUME3D_OT_BrowseScriptPath)
    bpy.utils.register_class(PLUME3D_OT_CreateNewScript)
    bpy.utils.register_class(PLUME3D_OT_ToggleTagLayer)
    bpy.utils.register_class(PLUME3D_OT_ToggleCollisionLayer)
    bpy.utils.register_class(PLUME3D_PT_WrenScriptConfigViewport)
    bpy.types.Object.plume3d_wren_config = PointerProperty(type=PLUME3D_WrenScriptConfig)
    bpy.types.Collection.plume3d_wren_config = PointerProperty(type=PLUME3D_WrenScriptConfig)
    if _sync_plume3d_to_id_properties not in bpy.app.handlers.save_pre:
        bpy.app.handlers.save_pre.append(_sync_plume3d_to_id_properties)
    if _sync_plume3d_to_id_properties not in bpy.app.handlers.load_post:
        bpy.app.handlers.load_post.append(_sync_plume3d_to_id_properties)
    bpy.types.WindowManager.plume3d_edit_collection = PointerProperty(
        name="Collection",
        type=bpy.types.Collection,
        description="Fallback: pick a collection when none is selected in the Outliner",
    )


def unregister():
    if _sync_plume3d_to_id_properties in bpy.app.handlers.save_pre:
        bpy.app.handlers.save_pre.remove(_sync_plume3d_to_id_properties)
    if _sync_plume3d_to_id_properties in bpy.app.handlers.load_post:
        bpy.app.handlers.load_post.remove(_sync_plume3d_to_id_properties)
    del bpy.types.WindowManager.plume3d_edit_collection
    del bpy.types.Object.plume3d_wren_config
    del bpy.types.Collection.plume3d_wren_config
    bpy.utils.unregister_class(PLUME3D_PT_WrenScriptConfigViewport)
    bpy.utils.unregister_class(PLUME3D_OT_ToggleCollisionLayer)
    bpy.utils.unregister_class(PLUME3D_OT_ToggleTagLayer)
    bpy.utils.unregister_class(PLUME3D_OT_CreateNewScript)
    bpy.utils.unregister_class(PLUME3D_OT_BrowseScriptPath)
    bpy.utils.unregister_class(PLUME3D_OT_CreateNewConfig)
    bpy.utils.unregister_class(PLUME3D_OT_BrowseGameConfigPath)
    bpy.utils.unregister_class(PLUME3D_OT_BrowseConfigPath)
    bpy.utils.unregister_class(PLUME3D_WrenScriptConfig)
    bpy.utils.unregister_class(PLUME3D_AddonPreferences)


if __name__ == "__main__":
    register()
